'==============================================================================
' Copyright  Micromega Corporation 2006-2007. All rights reserved.
'
' @file   GCdistance.bas
' @target PicBasic Pro
'
' see: Application Note 39 - Calculating Great Circle Distances
'
' This program demonstrates the calculation of great circle distances.
' It reads the current latitude and longitude position from a GPS receiver,
' then calculates the great circle distance to a number of stored locations.
' The program can also be run without a GPS connection by selecting one of
' the pre-defined locations as the 'from' location.
'
' @author Cam Thompson, Micromega Corporation
' @version
'	September 18, 2007
'	- modified for PicBasic Pro
'   July 25, 2007
'   - original version
'==============================================================================

'-------------------- PIC oscillator speed ------------------------------------
' The oscillator speed must be defined to ensure proper timing constants
' for the uM-FPU routines.

define      OSC 20                      ' specify the speed of the oscillator
OSC_SPEED   con 20                      ' define PicBasic constant

'-------------------- debug definitions ---------------------------------------
' The serial pin and baud rate for the DEBUG command are specified with the
' following defines. These should be changed to suit your application.

define      DEBUG_REG   PORTC
define      DEBUG_BIT   6
define      DEBUG_BAUD  19200
define      DEBUG_MODE  0

'-------------------- uM-FPU pin definitions ----------------------------------
' The following assignments define the pins used to connect to the uM-FPU V2.0
' chip. These should be changed to suit your application.

Fpu_SCL     var     PORTC.3             ' I2C SCL pin (uM-FPU SCL)
Fpu_SDA     var     PORTC.4             ' I2C SDA pin (uM-FPU SDA)
Fpu_ID      con     $C8                 ' uM-FPU I2C device address
define      I2C_HOLD    1               ' required for correct I2C timing

include     "umfpuV3-i2c.bas"           ' include the uM-FPU V3 support routines

'-------------------- uM-FPU Register Definitions -----------------------------
lat1            con     10              ' uM-FPU register 10
long1           con     11              ' uM-FPU register 11
lat2            con     12              ' uM-FPU register 12
long2           con     13              ' uM-FPU register 13

dist_nm         con     20              ' distance - nautical miles
dist_km         con     21              ' distance - kilometers
dist_miles      con     22              ' distance - miles

'-------------------- uM-FPU Function Definitions -----------------------------
getID           con     0               ' uM-FPU user function 0
getDistance     con     1               ' uM-FPU user function 1
getLocation     con     2               ' uM-FPU user function 2
getLatLong      con     3               ' uM-FPU user function 3
radiansToDM     con     4               ' uM-FPU user function 4
readNMEA        con     5               ' uM-FPU user function 5
parseGPRMC      con     6               ' uM-FPU user function 6
NMEA_Degrees    con     7               ' uM-FPU user function 7

'-------------------- variables -----------------------------------------------

from            var     Byte            ' from location index
index           var     Byte            ' to location index

'==============================================================================
'-------------------- initialization ------------------------------------------
'==============================================================================

Reset:
  DEBUG 13, 10, 13, 10, "Great Circle Distance", 13, 10

  GOSUB Fpu_Reset						' reset the FPU hardware
  IF fpu_status <> SYNC_CHAR THEN		' check for synchronization
    DEBUG "uM-FPU not detected"
    END
  ELSE
    GOSUB Print_Version					' display the uM-FPU version number
    DEBUG 13, 10
  ENDIF

  ' check that the user-defined functions are loaded
  I2CWRITE Fpu_SDA, Fpu_SCL, Fpu_ID, 0, [CLR0, FCALL, getID, SELECTA, 0]
  GOSUB Fpu_Wait
  I2CWRITE Fpu_SDA, Fpu_SCL, Fpu_ID, 0, [LREADBYTE]
  I2CREAD Fpu_SDA, Fpu_SCL, Fpu_ID, 0, [dataByte]
  IF dataByte <> 39 THEN
    DEBUG "GCdistance.fpu functions not loaded."
    END
  ENDIF

'==============================================================================
'-------------------- main routine --------------------------------------------
'==============================================================================

Main:

  ' if from = 99, read current location from GPS
  ' if from <> 99, use a stored location (0 to 7)
  ' ---------------------------------------------
  from = 99

  DEBUG 13, 10, "From:", 13, 10, "  "
  IF from = 99 THEN
    DEBUG "Reading GPS ..."
    I2CWRITE Fpu_SDA, Fpu_SCL, Fpu_ID, 0, [FCALL, readNMEA, STRSET, "Location", 0]
  ELSE
    I2CWRITE Fpu_SDA, Fpu_SCL, Fpu_ID, 0, _
      [LONGBYTE, from, FCALL, getLocation, COPY, lat2, lat1, COPY, long2, long1]
  ENDIF

  ' display latitude and longitude
  ' ------------------------------
  I2CWRITE Fpu_SDA, Fpu_SCL, Fpu_ID, 0, _
    [STRINS, " (", 0, SELECTA, lat1, FCALL, radiansToDM, STRINS, ", ", 0, _
     SELECTA, long1, FCALL, radiansToDM, STRINS, ")", 0]
  GOSUB Print_FpuString

  ' display distance to all other stored locations
  ' ----------------------------------------------
  DEBUG 13, 10, "To:"

  FOR index = 0 TO 7
    IF from <> index THEN

      DEBUG 13, 10, "  "
      I2CWRITE Fpu_SDA, Fpu_SCL, Fpu_ID, 0, _
        [LONGBYTE, index, FCALL, getLocation, _
         STRINS, " (", 0, SELECTA, lat2, FCALL, radiansToDM, STRINS, ", ", 0, _
         SELECTA, long2, FCALL, radiansToDM, STRINS, ")", 0]
      GOSUB Print_FpuString

      DEBUG 13, 10, "   "
      I2CWRITE Fpu_SDA, Fpu_SCL, Fpu_ID, 0, _
        [SELECTA, dist_nm, FCALL, getDistance, FSET0]
      format = 92
      GOSUB Print_FloatFormat
      DEBUG " nautical miles"

      DEBUG 13, 10, "   "
      I2CWRITE Fpu_SDA, Fpu_SCL, Fpu_ID, 0, _
        [SELECTA, dist_km, FSET, dist_nm, FWRITE0, $3F, $ED, $0E, $56, FMUL0, _
        SELECTA, dist_miles, FSET, dist_km, FCNV, 13]
      format = 92
      GOSUB Print_FloatFormat
      DEBUG " miles"

      DEBUG 13, 10, "   "
      I2CWRITE Fpu_SDA, Fpu_SCL, Fpu_ID, 0, [SELECTA, dist_km]
      format = 92
      GOSUB Print_FloatFormat
      DEBUG " kilometers"
    ENDIF
  NEXT

Done:
  DEBUG 13, 10, "Done.", 13, 10
  END
